#! /usr/bin/perl
#  Copyright 2001-2024 Leslie Richardson

#  This file is part of Open Admin for Schools.

#  Open Admin for Schools is free software; you can redistribute it 
#  and/or modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2 of 
#  the License, or (at your option) any later version.

my %lex = ('Main' => 'Main',
	   'Continue' => 'Continue',
	   'Homeroom' => 'Homeroom',
	   'Grade' => 'Grade',
	   'Select by' => 'Select by',
	   'Error' => 'Error',
	   'Sort by' => 'Sort by',
	   'Name' => 'Name',

	   'Common Math Assessment' => 'Common Math Assessment',
	   'Report' => 'Report',
	   'OR' => 'OR',
	   'No Selection' => 'No Selection',
	   'School Year' => 'School Year',
	   'Show Withdrawn' => 'Show Withdrawn',


	   );

my $self = 'cmaRpt4.pl';

use DBI;
use CGI;
use Cwd;
use Number::Format qw(:all);
use Time::JulianDay;

my @strands = qw(P N SS SP);
my %strandnames = ('P' => 'Patterns and Relations', 
		   'N' => 'Numbers and Operations',
		   'SS' => 'Shape and Space',
		   'SP' => 'Stats and Prob'
    );

=head
my %colormap = ( 1 => '#822',
		 2 => '#BB1',
		 3 => '#228',
		 4 => '#282'
    );
=cut

my %colormap = ( 1 => 'r',
		 2 => 'y',
		 3 => 'b',
		 4 => 'g'
    );


my  $configpath = '../../..';
if ( getcwd() =~ /tcgi/ ){ # we are in tcgi
    $configpath = '../..';
}

eval require "$configpath/etc/admin.conf";
if ( $@ ) {
    print $lex{Error}. " $@<br>\n";
    die $lex{Error}. " $@\n";
}

my $q = new CGI;
print $q->header( -charset, $charset );
my %arr = $q->Vars;


my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $iddst) = localtime(time);
$year = $year + 1900;
$wday++; $mon++;
my $currsdate = "$year-$mon-$mday";
my $currdate = "$dow[$wday], $month[$mon] $mday, $year";
my $currjd = julian_day( split('-', $currsdate) );

my $dsn = "DBI:$dbtype:dbname=$dbase";
my $dbh = DBI->connect($dsn,$user,$password);


# Get current dir so know what CSS to display;
if ( getcwd() =~ /tcgi/ ){ # we are in tcgi
    $css = $tchcss;
    $homepage = $tchpage;
}


my $title = "$lex{'Common Math Assessment'} $lex{Report} 4 - $schoolname";
print qq{$doctype\n<html><head><title>$title</title>\n};
print qq{<link rel="stylesheet" href="$css" type="text/css">\n};
print qq{$chartype\n</head><body style="padding:1em 5em;">\n};

print qq{<style>\n};
print qq{td.r { background-color:#822;color:white;font-size:120%;font-weight:bold;text-align:center;}\n};
print qq{td.y { background-color:#BB1;color:white;font-size:120%;font-weight:bold;text-align:center;}\n};
print qq{td.b { background-color:#228;color:white;font-size:120%;font-weight:bold;text-align:center;}\n};
print qq{td.g { background-color:#282;color:white;font-size:120%;font-weight:bold;text-align:center;}\n};
print qq{</style>\n};


print qq{<div>[ <a href="$homepage">$lex{Main}</a> ]</div>\n};

print qq{<h1>$title</h1>\n};


if ( not $arr{page} ) {
    showStartPage();

} elsif ( $arr{page} == 1 ) {
    delete $arr{page};
    showReport();
}


#-------------
sub showReport {
#-------------

    # foreach my $key ( sort keys %arr ) { print qq{K:$key V:$arr{$key}<br>\n}; }

    my $studenttable = 'student';
    if ( $arr{showwithdrawn} ) {
	$studenttable = 'studentwd';
    }

    
    if ( $arr{grade} and $arr{homeroom} ) {
	print qq{<h3>Select either Homeroom OR Grade, not both</h3>\n};
	print qq{</body></html>\n};
	exit;
    }
    

    my ($select, $selectval, @grades, $classname);
    if ( $arr{grade} ) {
	$select = "where grade = ?";
	$selectval = $arr{grade};
	push @grades, $arr{grade};
	$classname = "Grade $arr{grade} Students";

    } elsif ( $arr{homeroom} ) {
	$select = "where homeroom = ? and grade = ?";
	$selectval = $arr{homeroom};
	$classname = "Homeroom $arr{homeroom} Students";

	my $sth = $dbh->prepare("select distinct grade from student where homeroom = ? 
           and grade != '' and grade is not NULL order by grade");
	$sth->execute( $arr{homeroom} );
	if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
	while ( my $gr = $sth->fetchrow ) {
	    push @grades, $gr;
	}

    } else {
	print qq{<h3>$lex{'No Selection'}</h3>\n};
	print qq{</body></html>\n};
	exit;
    }


    # Figure out the number of days in the school year and also where
    # the current date fits into that.
    my %closed;    # $closed{julian date} = fraction of day closed.
    my $sth = $dbh->prepare("select date, dayfraction from dates");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    while ( my ($d, $frac) = $sth->fetchrow ) {
	if ( not $frac ) { next; } # just in case
	my $jd = julian_day( split('-', $d) );
	$closed{$jd} = $frac;
    }

    my $startjd = julian_day( split('-', $schoolstart) );
    my $endjd = julian_day( split('-', $schoolend) );

#    print qq{School Start: $schoolstart School End:$schoolend<br>\n};


    # loop through the school year; count # number of days until end and also until the current date.
    my ($totaldays, $partdays );
    for my $jd ( $startjd..$endjd ) { # run through the school year

	my $dow = day_of_week( $jd );
	if ( $dow == 6 or $dow == 1 ) { next; }

	if ( $closed{$jd} ) {
	    my $open = 1 - $closed{$jd};
	    $totaldays += $open;
	    if ( $jd <= $currjd ) { $partdays += $open; }

	} else {
	    $totaldays++;
	    if ( $jd <= $currjd ) { $partdays++; }
	}
    }


#    print qq{<br><br>Total:$totaldays Part:$partdays<br>\n};
 
    my $percentdone = format_number( $partdays * 100 / $totaldays, 1,1 );
    print qq{<h3>Percent of School Year Complete: $percentdone%</h3>\n};
    print qq{<h3>$classname  &ndash; $currdate</h3>\n};


    foreach my $grade ( @grades ) {
    
	# Build Outcomes selection
	my %outcomes; # lists outcomes.
	my %odesc; # outcomes description.

	my $sth = $dbh->prepare("select * from mathca_outcomes");
	$sth->execute;
	if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
	while ( my $ref = $sth->fetchrow_hashref ) {
	    my %r = %$ref;

	    my $oid = $r{oid};
	    my $grade = $r{grade};
	    #	print qq{OID:$oid<br>};

	    my ($cat, $seq)  = split(/\./, $oid);
	    $cat =~ s/$grade$//; # remove grade

	    # print qq{Grade:$grade CAT:$cat SEQ:$seq<br><br>\n};

	    $outcomes{$grade}{$cat}{$seq} = $oid;
	    $odesc{$oid} = $r{odesc};

	} # Done building data structure of outcomes

	
	my %reportgrade;  # based on students and school year.

	# We assume only one test grade for this school year
	my $sth1 = $dbh->prepare("select distinct tgrade from mathca_scores
	   where schoolyear = ? and studnum = ?");
	
	my $sth = $dbh->prepare("select studnum from $studenttable $select");
	# This is a duplicate of below but we want the students, and I don't want to mess up this script.
	if ( $arr{homeroom} ) { # add on grade selector to only get part of the class.
	    $sth->execute( $selectval, $grade );
	} else {
	    $sth->execute( $selectval );
	}
	if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
	while ( my $studnum = $sth->fetchrow ) {
	    $sth1->execute( $arr{schoolyear}, $studnum );
	    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
	    my $tgrade = $sth1->fetchrow;
	    #	    print qq{<div>$studnum $tgrade</div>\n};
	    if ( not $tgrade ) { next; }
	    $reportgrade{$tgrade} = 1;
	}

	my @temp = keys %reportgrade;
	my $rgrade = $temp[0]; # only grab first one. Should be no others.

	
	# Now build a structure for the grade for this reporting year
	# NOT necessarily the grade the students are now in.
	my %strandcount;  # $catcount{category} = number in this category.
	foreach my $cat ( sort keys %{ $outcomes{$rgrade} } ) {
	    foreach my $seq ( sort keys %{ $outcomes{$rgrade}{$cat} } ) {
		if ( $seq > $strandcount{$cat} ) { $strandcount{$cat} = $seq; }
	    }
	}

	my $totaloutcomes;
	foreach my $key ( keys %strandcount ) {
	    $totaloutcomes += $strandcount{$key};
	}

	
	# Now Select the students to display; skip any withdrawn kids w/o data.
	my $sth = $dbh->prepare("select lastname, firstname, studnum, grade from $studenttable
				$select order by lastname, firstname");
	if ( $arr{homeroom} ) { # add on grade selector to only get part of the class.
	    $sth->execute( $selectval, $grade );
	} else {
	    $sth->execute( $selectval );
	}
	if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

	
	my %withdrawn;
	my %data; # holds all data $data{studnum}{outcome}{prepost} = score;
	my @students;  # provides order of students
	my %pass; # $pass{studnum} = count of passes ( >= 3)
	my %strandpass; 

	my %names; # store student names;
	my %tdates; # store test dates for outcomes and pre/post  $tdates{outcome}{pretest/posttest};

	# Get Student Data
	my $sth1 = $dbh->prepare("select prepost, outcome, score, tdate from mathca_scores
          where schoolyear = ? and studnum = ? order by tdate");

	# Queries for Withdrawn and Having Data
	my $sth2 = $dbh->prepare("select count(*) from studentwd where studnum = ?");

	my $sth3 = $dbh->prepare("select count(*) from mathca_scores
          where schoolyear = ? and studnum = ?");



	# Loop through all students; build data structure.
	while ( my $ref = $sth->fetchrow_hashref ) {
	    my %r = %$ref;

	    my $studnum = $r{studnum};

	    
	    # Check for withdrawn and having any data this year.
	    if ( $arr{showwithdrawn} ) { # check if student is withdrawn
		# Withdrawn?
		$sth2->execute( $studnum );
		if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
		my $wdcount = $sth2->fetchrow;

		# Have Data?
		$sth3->execute($arr{schoolyear}, $studnum );
		if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
		my $datacount = $sth3->fetchrow;
		
		if ( $wdcount > 0 and $datacount < 1 ) { # withdrawn and no data
		    next;
		}
	    }


	    push @students, $studnum; 
	    $names{$r{studnum}} = "<b>$r{lastname}</b>, $r{firstname}";

#	    print qq{SN:$studnum Year:$arr{schoolyear}<br>\n};

	    # Get this student's data
	    $sth1->execute( $arr{schoolyear}, $studnum );
	    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

	    while ( my ($prepost, $outcome, $score, $tdate) = $sth1->fetchrow ) {
#		print qq{<div>SN:$studnum PREPOSt:$prepost OUT:$outcome SCR:$score Date:$tdate</div>\n};
		if ( $prepost eq 'posttest' ) {  # only do posttests for data.
		    push @{ $data{$studnum}{$outcome} }, $score;
		    if ( $score >= 3 ) { $pass{$studnum}{'pass'}++; }
		    $pass{$studnum}{'count'}++; 
		}
	    }
	}


	# Start Printing Data
	my $prev = $arr{schoolyear} - 1;
	print qq{<div style="font-size:120%;font-weight:bold;margin-top:1em;">};
	print qq{Reporting Grade $rgrade Year $prev-$arr{schoolyear}</div>\n};
	print qq{<table cellpadding="2" cellspacing="0" border="1">\n};
	print qq{<caption>Hover on Column Titles to see text descriptions</caption>\n};

	# header section.
	print qq{<tr><th>Student</th><th title="Pass Percentage">Pass<br>Per</th>};
	print qq{<th title="Mastery Level for Complete Curriculum">MST</th>};
	foreach my $strand ( @strands ) {
	    if ( $strandcount{$strand} ) { # print IF we have that strand
		print qq{<th colspan="$strandcount{$strand}" };
		print qq{title="Number in brackets is number of outcomes in this strand };
		print qq{for this reporting grade">};
		print qq{$strandnames{$strand} ($strandcount{$strand})</th>\n};
	    }
	}

	print qq{</tr>\n};

	# Second Line: Desc;
	print qq{<tr><th></th><th></th><th></th>};
	foreach my $strand ( @strands ) {
	    if ( $strandcount{$strand} ) { # print IF we have that strand
		foreach my $seq ( sort keys %{ $outcomes{$rgrade}{$strand} } ) {
		    my $outcome = $outcomes{$rgrade}{$strand}{$seq};
		    print qq{<th title="$odesc{$outcome}">$outcome</th>};
		}
	    }
	}
	print qq{</tr>\n};


	# Loop through each student
	foreach my $studnum ( @students ) {

	    print qq{<tr><td>$names{$studnum}</td>\n};

	    # %pass
	    my ($pass, $mst, $outcomecount);
	    if ( $pass{$studnum}{count} ) {
		$pass = format_number( $pass{$studnum}{pass} * 100 / $pass{$studnum}{count}, 0, 0);
		#if ( $pass ) { $pass .= '%'; }
	    }

	    # now figure out how many outcomes have scores
	    foreach my $outcome ( keys %{ $data{$studnum} } ) {
		if ( ${ $data{$studnum}{$outcome} }[0] ) {
		    $outcomecount++;
		}
	    }
	    if ( $pass{$studnum}{count} and $totaloutcomes ) {
		$mst = format_number( $pass{$studnum}{pass} /
				      $pass{$studnum}{count} * $outcomecount / $totaloutcomes,  2, 2);
	    }

	    print qq{<td>$pass</td><td>$mst</td>};

	    foreach my $strand ( @strands ) {

		if ( not $strandcount{$strand} ) { next; } # skip any strands not covered in this reporting grade.
		
		foreach my $seq ( sort keys %{ $outcomes{$rgrade}{$strand} } ) {
		    my $outcome = $outcomes{$rgrade}{$strand}{$seq};
		    
		    my @scores = @{ $data{$studnum}{$outcome} };

		    my $displayval;
		    my $first = 1;
		    my $max;

		    foreach my $score ( @scores ) {
			if ( not $first ) { 
			    $displayval .= qq{/}; 
			} else { 
			    $first = 0; 
			}
			$displayval .= $score;
			if ( $score > $max ) { $max = $score; }

		    }

		    my $class = $colormap{$max};
		    
		    print qq{<td class="$class">$displayval</td>};

		}

	    } # end of this strand


=head
	# now totals for this student
	my ($pretotalavg, $posttotalavg);
	if ( $pretotalcount ) {
	    $pretotalavg = format_number( $pretotalsum / $pretotalcount, 2);
	}

	if ( $posttotalcount ) {
	    $posttotalavg = format_number( $posttotalsum / $posttotalcount, 2);
	}

	my $diff = format_number( $posttotalavg - $pretotalavg, 2);

	print qq{<td class="cn">$posttotalcount</td><td class="cn">$pretotalavg</td>};
	print qq{<td class="cn">$posttotalavg</td><td class="cn">$diff</td>\n};

	# Passes;
	my $passpercent;
	if ( $pass{$studnum}{'count'} and $pass{$studnum}{'pass'} ) {
	    $passpercent = format_number( $pass{$studnum}{'pass'} * 100 / $pass{$studnum}{'count'}, 2);
	}
	
	if ( $passpercent ) {
	    print qq{<td>$passpercent% ($pass{$studnum}{'pass'}/$pass{$studnum}{'count'})</td></tr>\n};
	} elsif ( $pass{$studnum}{'count'} ) { # have count, but no passes
	    print qq{<td>0% (0/$pass{$studnum}{'count'})</td></tr>\n};
	} else { # print blank.
	    print qq{<td></td></tr>\n};
	}
=cut


	} # end of student loop

	print qq{</table>\n};


    } # next grade.

    print qq{<div style="padding:1em 1em 0 1em;font-weight:bold;">};
    print qq{All scores are posttest scores. Pretest scores are };
    print qq{not shown on this report</div>\n};

    print qq{<div style="padding:1em 1em 0 1em;font-weight:bold;">};
    print qq{If more than one test is given to a student on the };
    print qq{same outcome, these scores are separated by slashes</div>\n};

    print qq{</body></html>\n};

    exit;

} # end of showReport



#----------------
sub showStartPage { # Entry Values for Custom Script
#----------------

    my (@homerooms, @grades, @schoolyears );
    # Get Homerooms
    my $sth = $dbh->prepare("select distinct homeroom from student 
      where homeroom is not NULL and homeroom != ''");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    while ( my $hr = $sth->fetchrow ) {
	push @homerooms, $hr;
    }
    @homerooms = sort {$a <=> $b} @homerooms;

    # Get Grades
    $sth = $dbh->prepare("select distinct grade from student 
      where grade is not NULL and grade != ''");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    while ( my $gr = $sth->fetchrow ) {
	push @grades, $gr;
    }
    @grades = sort {$a <=> $b} @grades;

    # Get School Years
    $sth = $dbh->prepare("select distinct schoolyear from mathca_scores");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    while ( my $yr = $sth->fetchrow ) {
	push @schoolyears, $yr;
    }
    @schoolyears = reverse sort @schoolyears;


    print qq{<form action="$self" method="post">\n};
    print qq{<input type="hidden" name="page" value="1">\n};


    # print qq{<div><input type="submit" value="$lex{Continue}"></div>\n};

    print qq{<table cellpadding="3" cellspacing="0" border="0">\n};


    # Select Grade
    print qq{<tr><td class="bra">$lex{'Select by'} $lex{Grade}</td>};
    print qq{<td><select name="grade"><option></option>\n};
    foreach my $gr ( @grades ) {
	print qq{<option>$gr</option>\n};
    }
    print qq{</select></td></tr>\n};

    # OR
    print qq{<tr><td colspan="2" style="text-align:center;">$lex{OR}</td></tr>\n};


    # Select Homeroom
    print qq{<tr><td class="bra">$lex{'Select by'} $lex{Homeroom}</td>};
    print qq{<td><select name="homeroom"><option></option>\n};
    foreach my $hr ( @homerooms ) {
	print qq{<option>$hr</option>\n};
    }
    print qq{</select></td></tr>\n};


    # School Year
    print qq{<tr><td class="bra">Reporting $lex{'School Year'}</td>};
    print qq{<td><select name="schoolyear">\n};
    foreach my $yr ( @schoolyears ) {
	my $prevyr = $yr - 1;
	print qq{<option value="$yr">$prevyr-$yr</option>\n};
    }
    print qq{</select></td></tr>\n};

    # Withdrawn
    print qq{<tr><td class="bra">$lex{'Show Withdrawn'}</td>};
    print qq{<td><input type="checkbox" name="showwithdrawn" value="1"></td></tr>\n};

    print qq{<tr><td class="ra"><input type="submit" value="$lex{Continue}"></td></tr>\n};
    print qq{</table>\n};
    print qq{</form></body></html>\n};

    exit;

}

