#!/usr/bin/perl
#  Copyright 2001-2024 Leslie Richardson

#  This file is part of Open Admin for Schools.

#  Open Admin for Schools is free software; you can redistribute it 
#  and/or modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2 of 
#  the License, or (at your option) any later version.

my %lex = ('Main' => 'Main',
	   'Report Card' => 'Report Card',
	   'Select a Student' => 'Select a Student',
	   'Select a Record' => 'Select a Record',
	   'Save Record' => 'Save Record',
	   'Edit Data Record' => 'Edit Data Record',
	   'Record Updated' => 'Record Updated',
	   'Edit Transcript Records' => 'Edit Transcript Records',
	   'Record Deleted' => 'Record Deleted',
	   'New Student Edit' => 'New Student Edit',
	   'Delete the Record' => 'Delete the Record',
	   'Confirm Record Delete' => 'Confirm Record Delete',
	   'Code' => 'Code',
	   'Subject' => 'Subject',
	   'Area' => 'Area',
	   'Term' => 'Term',
	   'Grade' => 'Grade',
	   'Difficulty' => 'Difficulty',
	   'School Year' => 'School Year',
	   'Description' => 'Description',
	   'Date' => 'Date',
	   'Credit' => 'Credit',
	   'Mark' => 'Mark',
	   'Edit' => 'Edit',
	   'Delete' => 'Delete',
	   'Error' => 'Error',
	   'Student Number' => 'Student Number',
	   'First Name' => 'First Name',
	   'Middle Name' => 'Middle Name',
	   'Last Name' => 'Last Name',
	   'Birthdate' => 'Birthdate',
	   'Student Id' => 'Student Id',
	   'Edit Identity' => 'Edit Identity',
	   'Graduation Date' => 'Graduation Date',
	   'Letter' => 'Letter',

	   );

my $self = 'tscdeled.pl';

use DBI;
use CGI;

eval require "../../etc/admin.conf";
if ( $@ ) {
    print $lex{Error}. " $@<br>\n";
    die $lex{Error}. " $@\n";
}

my $q = CGI->new;
print $q->header( -charset, $charset );
my %arr = $q->Vars;

my $dsn = "DBI:$dbtype:dbname=$dbase";
my $dbh = DBI->connect($dsn,$user,$password);


# Page Header
my $title = $lex{'Edit Transcript Records'};
print qq{$doctype\n<html><head><title>$title</title>
<link rel="stylesheet" href="$css" type="text/css">
$chartype\n</head><body>\n};

print qq{[ <a href="$homepage">$lex{Main}</a> |\n};
print qq{<a href="$reppage">$lex{'Report Card'}</a> ]\n};

if ( $arr{studid} ) { # We have a selected student.
    showSelectedStudent( $arr{studid} );

} elsif ( $arr{dataid} ) {
    editDataRecord( $arr{dataid} );

} elsif ( $arr{identid} ) {
    editIdentRecord( $arr{identid} );

} elsif ( $arr{writeidentflag} ) {
    delete $arr{writeidentflag};
    writeIdentRecord();

} elsif ( $arr{delid} ) {
    confirmDataDelete( $arr{delid} );

} elsif ( $arr{writedataflag} ) {
    delete $arr{writedataflag};
    writeDataRecord();

} elsif ( $arr{deletedataflag} ) {
    delete $arr{deletedataflag};
    deleteDataRecord( $arr{id} );
}


# Student Selection
my $sth = $dbh->prepare("select id, studnum, lastname, middlename, firstname from tscriptident 
  order by lastname, firstname");
$sth->execute;
if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

# Print Start of Selection.
print qq{<h1>$title</h1>\n};
print qq{<table cellpadding="3" border="1" cellspacing="0">};
print qq{<tr><th colspan="3">$lex{'Select a Student'}</th></tr>\n};


my $count = 1;
print qq{<tr>};
while ( my ( $id, $studnum, $lastname, $middlename, $firstname ) = $sth->fetchrow ) {
    print qq{<td><a href="$self?studid=$id"><strong>$lastname</strong>, };
    print qq{$firstname $middlename</a> ($studnum)</td>\n};
    if (not ($count % 3)){ print qq{</tr>\n<tr>};}
    $count++;
}
if ($count % 3 == 2){ print qq{<td></td><td></td></tr>\n}; }
if ($count % 3 == 0){ print qq{<td></td></tr>\n}; }

print qq{</table></body></html>\n};



#----------------------
sub showSelectedStudent {
#----------------------

    my $id = shift;

    # Get Student Ident 
    my $sth = $dbh->prepare("select studnum, lastname, middlename, firstname 
     from tscriptident where id = ?");
    $sth->execute( $id );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my ($studnum, $lastname, $middlename, $firstname) = $sth->fetchrow;

    # Get Student Data
    $sth = $dbh->prepare("select id, subjecttext, subjectcode, subjectarea, schoolyear, term_desc 
     from tscriptdata where studnum = ? order by schoolyear, term");
    $sth->execute( $studnum );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

    print qq{<h1 style="display:inline;">$lex{'Edit Transcript Records'} &ndash; };
    print qq{$firstname $middlename $lastname</h1>\n};

    print qq{<form action="$self" method="post" style="display:inline;">\n};
    print qq{<input type="submit" value="$lex{'Edit Identity'}">\n};
    print qq{<input type="hidden" name="identid" value="$id"></form>\n};


    print qq{<table cellpadding="3" border="1" cellspacing="0">};
    print qq{<tr><th colspan="5">$lex{'Select a Record'}</th></tr>\n};

    while (my ( $id, $subjectcode, $subjecttext, $subjectarea, $schoolyear, $term_desc ) 
	   = $sth->fetchrow ) {
	print qq{<tr><td>$subjecttext ($subjectcode)</td><td>$subjectarea</td>};
	print qq{<td>$schoolyear - $term_desc</td>};
	print qq{<td><a href="$self?dataid=$id">$lex{Edit}</a>\n};
	print qq{</td><td>};
	print qq{<a href="$self?delid=$id">$lex{Delete}</a>};
	print qq{</td></tr>\n};
    }

    print qq{</table></body></html>\n};

    exit;

}


#-----------------
sub editIdentRecord {
#-----------------

    my $identid = shift;

    # Get Ident Record
    my $sth = $dbh->prepare("select * from tscriptident where id = ?");
    $sth->execute( $identid );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my ($id,$studnum, $lastname, $firstname, $middlename,
	$birthdate, $studentid, $graddate ) = $sth->fetchrow;
    # Note: $studentid may be a state or provincial identity number. NOT studnum (local).
    # The field is set in the transcript posting script.

    print qq{<h1>$lex{'Edit Identity'} &mdash; $firstname $lastname</h1>\n};

    # Now print the form.
    print qq{<form action="$self" method="post">\n};
    print qq{<input type="hidden" name="id" value="$id">\n};
    print qq{<input type="hidden" name="writeidentflag" value="1">\n};
    print qq{<input type="hidden" name="studnum" value="$studnum">\n};
    print qq{<table cellpadding="3" cellspacing="0" border="0">\n};
    print qq{<tr><td></td><td><input type="submit" value="$lex{'Save Record'}"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{'Student Number'}</td>\n<td class="la">};
    print qq{$studnum</td></tr>\n};

    print qq{<tr><td class="bra">$lex{'First Name'}</td>\n<td class="la">};
    print qq{<input type="text" name="firstname" size="20" value="$firstname"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{'Middle Name'}</td>\n<td class="la">};
    print qq{<input type="text" name="middlename" size="20" value="$middlename">};
    print qq{</td></tr>\n};

    print qq{<tr><td class="bra">$lex{'Last Name'}</td>\n<td class="la">};
    print qq{<input type="text" name="lastname" size="20" value="$lastname"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{'Birthdate'}</td>\n<td class="la">};
    print qq{<input type="text" name="birthdate" size="13" value="$birthdate"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{'Student Id'}</td>\n<td class="la">};
    print qq{<input type="text" name="studentid" size="13" value="$studentid"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{'Graduation Date'}</td>\n<td class="la">};
    print qq{<input type="text" name="graddate" size="13" value="$graddate"></td></tr>\n};

    print qq{<tr><td></td><td><input type="submit" value="$lex{'Save Record'}"></td></tr>\n};

    print qq{</table></form></body></html>\n};


    exit;

} # End of editIdentRecord


#------------------
sub writeIdentRecord {
#------------------

    my $studid = $arr{id}; # for passing value below
    foreach my $key (keys %arr ) { 
	$arr{$key} = $dbh->quote( $arr{$key} );
	#print qq{K:$key V:$arr{$key}<br>\n}; 
    }

    my $sth = $dbh->prepare("update tscriptident set
       firstname = $arr{firstname},
       middlename = $arr{middlename},
       lastname = $arr{lastname},
       birthdate = $arr{birthdate},
       studentid = $arr{studentid},
       graddate = $arr{graddate}
     where id = $arr{id} ");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

    print qq{<p><b>$lex{'Record Updated'}</b></p>\n};

    print qq{[ <a href="$self?studid=$studid">$lex{'Edit Transcript Records'}</a> |\n};
    print qq{ <a href="$self">$lex{'New Student Edit'}</a> ]\n};

    print qq{</body></html>\n};

    exit;

} # end of writeIdentRecord




#-----------------
sub editDataRecord {
#-----------------

    my $dataid = shift;

    require "../../etc/transcript.conf" or die "Cannot open transcript.conf!";

    # Get Data Record
    my $sth = $dbh->prepare("select * from tscriptdata where id = ?");
    $sth->execute( $dataid );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my ($id,$studnum, $subjectcode, $subjecttext, $subjectarea, 
	$score_mark, $score_letter, $score_diff, $schoolyear, $crdate, $credit, 
	$term, $term_desc ) = $sth->fetchrow;

    $sth = $dbh->prepare("select lastname, firstname from tscriptident where studnum = ?");
    $sth->execute( $studnum );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my ($lastname, $firstname) = $sth->fetchrow;


    print qq{<h1>$lex{'Edit Data Record'} &mdash; $firstname $lastname</h1>\n};


    # Now print the form.
    print qq{<form action="$self" method="post">\n};
    print qq{<input type="hidden" name="id" value="$id">\n};
    print qq{<input type="hidden" name="writedataflag" value="1">\n};
    print qq{<input type="hidden" name="studnum" value="$studnum">\n};

    print qq{<table cellpadding="3" cellspacing="0" border="0">\n};
    print qq{<tr><td></td><td><input type="submit" value="$lex{'Save Record'}"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Subject} $lex{Code}</td>\n<td class="la">};
    print qq{<input type="text" name="subjectcode" size="8" value="$subjectcode"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Subject}</td>\n<td class="la">};
    print qq{<input type="text" name="subjecttext" size="25" value="$subjecttext"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Subject} $lex{Area}</td>\n<td class="la">};


    if ( %gradRequirements ) {
	print qq{<select name="subjectarea"><option>$subjectarea</option>\n};
	foreach my $key ( sort keys %gradRequirements ) {
	    print qq{<option>$key</option>};
	}
	print qq{<option></option></select></td></tr>\n};
    } else {
	print qq{<input type="text" name="subjectarea" size="15" value="$subjectarea"></td></tr>\n};
    }


    print qq{<tr><td class="bra">$lex{Mark}</td>\n<td class="la">};
    print qq{<input type="text" name="score_mark" size="4" value="$score_mark"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Letter} $lex{Grade}</td>\n<td class="la">};
    print qq{<input type="text" name="score_letter" size="4" value="$score_letter"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Difficulty}</td>\n<td class="la">};
    print qq{<input type="text" name="score_diff" size="4" value="$score_diff"></td></tr>\n};


    print qq{<tr><td class="bra">$lex{'School Year'}</td>\n<td class="la">};
    print qq{<input type="text" name="schoolyear" size="10" value="$schoolyear"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Date}</td>\n<td class="la">};
    print qq{<input type="text" name="crdate" size="13" value="$crdate"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Credit}</td>\n<td class="la">};
    print qq{<input type="text" name="credit" size="4" value="$credit"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Term}</td>\n<td class="la">};
    print qq{<input type="text" name="term" size="4" value="$term"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Term} $lex{Description}</td>\n};
    print qq{<td class="la">};
    print qq{<input type="text" name="term_desc" size="15" value="$term_desc"></td></tr>\n};

    print qq{<tr><td></td><td><input type="submit" value="$lex{'Save Record'}"></td></tr>\n};

    print qq{</table></form></body></html>\n};


    exit;


}


#------------------
sub writeDataRecord {
#------------------

    foreach my $key (keys %arr ) { 
	$arr{$key} = $dbh->quote( $arr{$key} );
	#print qq{K:$key V:$arr{$key}<br>\n}; 
    }

    my $sth = $dbh->prepare("update tscriptdata set
       subjectcode = $arr{subjectcode},
       subjecttext = $arr{subjecttext},
       subjectarea = $arr{subjectarea},
       score_mark  = $arr{score_mark},
       score_letter  = $arr{score_letter},
       score_diff  = $arr{score_diff},
       schoolyear = $arr{schoolyear},
       crdate = $arr{crdate},
       credit = $arr{credit},
       term = $arr{term},
       term_desc = $arr{term_desc}
     where id = $arr{id} ");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

    print qq{<p><b>$lex{'Record Updated'}</b></p>\n};

    # Now get student identid for record
    $sth = $dbh->prepare("select id from tscriptident where studnum = $arr{studnum}");
    $sth->execute;
    my $studid = $sth->fetchrow;

    print qq{[ <a href="$self?studid=$studid">$lex{'Edit Transcript Records'}</a> |\n};
    print qq{ <a href="$self">$lex{'New Student Edit'}</a> ]\n};

    print qq{</body></html>\n};

    exit;

}



#------------------
sub deleteDataRecord {
#------------------

    my $id = shift;

    foreach my $key (keys %arr ) { print qq{K:$key V:$arr{$key}<br>\n}; }

    # Get student ident record id from data record id (2 steps).
    my $sth = $dbh->prepare("select studnum from tscriptdata where id = ?");
    $sth->execute( $id );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my $studnum = $sth->fetchrow;

    $sth = $dbh->prepare("select id from tscriptident where studnum = ?");
    $sth->execute( $studnum );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my $studid = $sth->fetchrow;

    # Delete the data record.
    $sth = $dbh->prepare("delete from tscriptdata where id = ?");
    $sth->execute( $id );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }

    # Response
    print qq{<h1>$lex{'Record Deleted'}</h1>\n};

    print qq{[ <a href="$self?studid=$studid">$lex{'Edit Transcript Records'}</a> |\n};
    print qq{ <a href="$self">$lex{'New Student Edit'}</a> ]\n};
    print qq{</body></html>\n};

    exit;

}



#------------------
sub confirmDataDelete {
#------------------

    my $id = shift;

    #foreach my $key (keys %arr ) { print qq{K:$key V:$arr{$key}<br>\n}; }

    # Get Data Record
    my $sth = $dbh->prepare("select * from tscriptdata where id = ?");
    $sth->execute( $id );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my ($id,$studnum, $subjectcode, $subjecttext, $subjectarea, 
	$score_mark, $score_letter, $score_diff, $schoolyear, $crdate, $credit, 
	$term, $term_desc ) = $sth->fetchrow;

    # Get Student Info
    $sth = $dbh->prepare("select id,lastname, firstname from tscriptident where studnum = ?");
    $sth->execute( $studnum );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my ($studid, $lastname, $firstname) = $sth->fetchrow;


    print qq{<h1>$lex{'Confirm Record Delete'} &mdash; $firstname $lastname</h1>\n};


    # Now print the form.
    print qq{<form action="$self" method="post">\n};
    print qq{<input type="hidden" name="id" value="$id">\n};
    print qq{<input type="hidden" name="deletedataflag" value="1">\n};
    print qq{<input type="hidden" name="studnum" value="$studnum">\n};

    print qq{<table cellpadding="3" cellspacing="0" border="0">\n};
    print qq{<tr><td></td><td class="la"><input type="submit" value="$lex{'Delete the Record'}"></td></tr>\n};

    print qq{<tr><td class="bra">$lex{Subject} $lex{Code}</td>\n<td class="la">};
    print qq{$subjectcode</td></tr>\n};

    print qq{<tr><td class="bra">$lex{Subject}</td>\n<td class="la">};
    print qq{$subjecttext</td></tr>\n};

    print qq{<tr><td class="bra">$lex{Subject} $lex{Area}</td>\n<td class="la">};
    print qq{$subjectarea</td></tr>\n};


    print qq{<tr><td class="bra">$lex{Mark}</td>\n<td class="la">};
    print qq{$score_mark</td></tr>\n};

    print qq{<tr><td class="bra">$lex{Letter} $lex{Grade}</td>\n<td class="la">};
    print qq{$score_letter</td></tr>\n};

    print qq{<tr><td class="bra">$lex{Difficulty}</td>\n<td class="la">};
    print qq{$score_diff</td></tr>\n};


    print qq{<tr><td class="bra">$lex{'School Year'}</td>\n<td class="la">};
    print qq{$schoolyear</td></tr>\n};

    print qq{<tr><td class="bra">$lex{Term} $lex{Description}</td>\n<td class="la">};
    print qq{$term_desc</td></tr>\n};

    print qq{<tr><td></td><td class="la"><input type="submit" value="$lex{'Delete the Record'}"></td></tr>\n};

    print qq{</table></form>\n};

    print qq{[ <a href="$self?studid=$studid">$lex{'Edit Transcript Records'}</a> |\n};
    print qq{ <a href="$self">$lex{'New Student Edit'}</a> ]\n};
    print qq{</body></html>\n};

    exit;

}

