#!/usr/bin/perl
#  Copyright 2001-2024 Leslie Richardson

#  This file is part of Open Admin for Schools.
#  Open Admin for Schools is free software; you can redistribute it 
#  and/or modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2 of 
#  the License, or (at your option) any later version.

%lex = ('Edit' => 'Edit',
	'Staff Member' => 'Staff Member',
	'Main' => 'Main',
	'Eoy' => 'Eoy',
	'Unable to open template file:' => 'Unable to open template file:',
	'Contact' => 'Contact',
	'Update' => 'Update',
	'Error' => 'Error',
	'Password' => 'Password',
	'You MUST have a unique userid and a password' => 
	  'You MUST have a unique userid and a password',
	'Grades' => 'Grades',
	'Homeroom(s)' => 'Homeroom(s)',
	'Position' => 'Position',
	'Record(s) Updated' => 'Record(s) Updated',
	'Separate with Spaces' => 'Separate with Spaces',
	'Add' => 'Add',
	'Required' => 'Required',
	'Bold' => 'Bold',
	'Preregistration' => 'Preregistration',
	'Access Control(s)' => 'Access Control(s)',
	'Current' => 'Current',
	'Spaces Not Allowed' => 'Spaces Not Allowed',
	'Type' => 'Type',
	'Percent' => 'Percent',
	'Occupation' => 'Occupation',
	'Userid Already Exists' => 'Userid Already Exists',
	'Staff' => 'Staff',
	
	 );


my $self = 'staffed.pl';
my @grades = qw(P3 PK K 1 2 3 4 5 6 7 8 9 10 11 12);

use CGI;
use DBI;
use Data::Password qw(:all);  # for password checking...
use Crypt::GeneratePassword qw(:all); # password generation.


# First Nations settings.
my $maxmulti = 4; # Occupation Limits for INAC First Nations occupations.
my $template = 'staff.tpl';
my $fnmode; # set below configuration load.



# Password config; rest in admin.conf
# For Generation
$g_staffpwd_minfreq = .001;
$g_staffpwd_avgfreq = .001;
$g_staffpwd_lang = 'en'; # only en or de available.


# For Checking
my $g_staffpwd_groups = 0; # turn off character group (uppercase, lowercase, symbols) checking
my $g_staffpwd_following = 0; # turn off following character checking (keyboard, same)

# WATCH: Only loading userids and passwords here!
eval require "../../etc/admin.conf";
if ( $@ ) {
    print $lex{Error}. ": $@<br>\n";
    die $lex{Error}. ": $@\n";
}

eval require "../../lib/libmeta.pl";
if ( $@ ) {
    print $lex{Error}. " $@<br>\n";
    die $lex{Error}. " $@\n";
}

# Load audit write function
eval require "../../lib/libaudit.pl";
if ( $@ ) {
    print $lex{Error}. ": $@<br>\n";
    die $lex{Error}. ": $@\n";
}


# Set Table to edit/add
my $table;
if ( $arr{tbl} ) {
    $table = 'prereg_staff';
    $multitable = 'prereg_staff_multi';
} else {
    $table = 'staff';
    $multitable = 'staff_multi';
}


my $dsn = "DBI:$dbtype:dbname=$dbase";
my $dbh = DBI->connect($dsn,$user,$password);


my $q = new CGI;
print $q->header(-charset, $charset ); 
my %arr = $q->Vars;
my $recid = $arr{id};


if ( $g_InacSchool ) { # if no mode here, normal editing operations.
    $fnmode = 1;  # first nation mode, additional fields, different template
    $template = 'staff_inac.tpl';
}


# Print Page Header
my $title = qq{$lex{Add}/$lex{Edit} $lex{'Staff Member'}};
print qq{$doctype\n<html><head><title>$title</title>\n};
print qq{<link rel="stylesheet" href="$css" type="text/css">\n};

print qq{$chartype\n</head><body>\n};
print qq{[ <a href="$homepage">$lex{Main}</a> | };
print qq{<a href="$eoypage">$lex{Eoy}</a> ]\n};

print qq{<h1>};
if ($arr{tbl}) {
    print qq{$lex{Edit} <span style="color:red;">$lex{Preregistration}</span> $lex{'Staff Member'}};
} else {
    print qq{$title};
}
print qq{</h1>\n};

if ( $fnmode ) {
    print qq{<div>First Nations Mode - INAC School Number $g_InacSchool</div>\n};
}


if ( $arr{writeflag} ) { # Write the records
    delete $arr{writeflag};
    writeRecord();
}

print qq{<p><b>$lex{Bold}</b> = $lex{Required}</p>\n};


# Load the record into a hash, if an edit process.
my $sth = $dbh->prepare("select * from staff where id = ?");
$sth->execute( $arr{id} );
if ($DBI::errstr){ print $DBI::errstr; die $DBI::errstr;}

# Test Loaded Values
#my @rec = $sth->fetchrow;
#print qq{@rec<br>\n};
#exit;

my $ref = $sth->fetchrow_hashref;
my %rec = %$ref;


# Read in Template as single text variable
unless (open (FH,"<../../template/$template")) {
    print qq{$lex{'Unable to open template file:'} - $!\n};
    die $lex{'Unable to open template file:'},"$!\n";
}
my $formtext;
{ local $/; $formtext = <FH>; close FH;}


# Create hash for fieldnames from meta.
$sth = $dbh->prepare("select fieldid, fieldname, required 
  from meta where tableid = ?");
$sth->execute('staff');
if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
my %fieldnames;
while ( my ( $fieldid, $fieldname, $required ) = $sth->fetchrow ) {
    if ( $required ) { # has any value
	$fieldname = qq{<span style="font-weight:bold;">$fieldname</span>};
    }
    $fieldnames{$fieldid} = $fieldname;
}

# Now put replacement text back in.
$formtext =~ s{\<\*(.*?)\*\>}
  { exists( $fieldnames{$1} ) 
	? $fieldnames{$1} 
	: $1
  }gsex;

# Find all fields , so we only wrap forms around them (typically
# faster than doing all fields in the table )
my @fields;
while ( $formtext =~ m/\<\@(.*)\@\>/g){
    push @fields, $1;
}


# Now set password, other config values at top of script
if ( not $rec{password} ) {
    my $pwd = word( $g_staffpwd_minlen, $g_staffpwd_genlen,
		    $g_staffpwd_lang, $g_staffpwd_signs,
		    $g_staffpwd_caps, $g_staffpwd_minfreq,
		    $g_staffpwd_avgfreq );


    $rec{password} = $pwd;
}


# get replacement values for fields, %rec holds values
my %values;
foreach my $fieldid ( @fields ) {
    $values{$fieldid} = metaInputField(
	'staff', $fieldid, $rec{ $fieldid }, $dbh,'' );
}

# now put field values back into $text variable...
$formtext =~ s{ \<\@(.*?)\@\> }
  { exists($values{$1}) 
	? $values{$1} 
	: "$values{$1}-$1"
  }gsex;


# print top of form, then $text, then bottom of page.
print qq{<form action="$self" method="post">\n};
print qq{<input type="hidden" name="writeflag" value="1">\n};
print qq{<input type="hidden" name="tbl" value="$arr{tbl}">\n};
print qq{<input type="hidden" name="id" value="$recid">\n};

print qq{<div style="margin:1em;">};
print qq{<input type="submit" value="$lex{Update} $lex{'Staff Member'}"></div>\n};

print $formtext,"\n";



# Teaching Positions (classroom, special ed, culture, principal, etc.)
$sth = $dbh->prepare("select id, field_value from $multitable 
  where field_name = 'position' and userid = ?");
$sth->execute( $rec{userid} );
if ($DBI::errstr){ print $DBI::errstr; die $DBI::errstr;}

print qq{<table cellpadding="3" cellspacing="0" border="0" };
print qq{style="margin:1em;padding:0.4em;border:1px solid gray;margin-bottom:1em;">\n};
print qq{<caption style="font-weight:bold;text-align:left;">Teaching Positions</caption>\n};
my $count = 1;

while ( my ( $id, $fieldvalue ) = $sth->fetchrow ) {
    print qq{<tr><td class="ra">$lex{Staff} $lex{Position} $count</td><td><select name="m:$id">};
    print qq{<option>$fieldvalue</option><option></option>\n};
    foreach my $pos ( @g_position ) { print qq{<option>$pos</option>}; }
    print qq{\n</select></td></tr>\n};
    $count++;
}

print qq{<tr><td class="ra">$lex{Add} $lex{Position}</td><td><select };
print qq{name="m:addposition"><option></option>\n};

foreach my $pos ( @g_position ) { print qq{<option>$pos</option>}; }
print qq{\n</select></td></tr>\n};
print qq{</table>\n};



# Grades for Homeroom and Nomroll special teachers.
print qq{<table cellpadding="3" cellspacing="0" border="0" };
print qq{style="margin:1em;padding:0.4em;border:1px solid gray;">\n};
print qq{<caption style="font-weight:bold;text-align:left;">};
print qq{Grades - Homeroom (K-9) and <br>Nominal Roll (Spec Ed, Cultural, Guidance)</caption>\n};

# print qq{<tr><td colspan="2" style="background-color:#888;color:white;">};
# print qq{Match with Student Grade values</td></tr>\n};

my $sth = $dbh->prepare("select id, field_value from $multitable 
  where field_name = 'grade' and userid = ? order by field_value");
$sth->execute( $rec{userid} );
if ($DBI::errstr){ print $DBI::errstr; die $DBI::errstr;}
print qq{<tr style="background-color:#DDD;"><td class="ra">$lex{Current} $lex{Grades} </td>\n<td>};

while ( my ( $id, $fieldvalue ) = $sth->fetchrow ) {
    print qq{ <input type="text" name="m:$id" style="width:6ch;" value="$fieldvalue">};
}
print qq{</td></tr>\n};

print qq{<tr><td class="ra">$lex{Add} $lex{Grades}</td><td>};
foreach my $idx (1..3) {
    print qq{<select name="m:addgrade$idx"><option></option>\n};
    foreach my $g (@grades) { print qq{<option>$g</option>}; }
    print qq{</select>\n};
}

#<input type="text" };
# print qq{name="m:addgrade" style="width:12ch;"> $lex{'Separate with Spaces'}
print qq{</td></tr>\n};

print qq{</table>\n};



# Homeroom 
print qq{<table cellpadding="3" cellspacing="0" border="0" };
print qq{style="margin:1em;padding:0.4em;border:1px solid gray;">\n};
print qq{<caption style="font-weight:bold;text-align:left;">};
print qq{Homeroom (typically PK-9)</caption>\n};

my $sth = $dbh->prepare("select id, field_value from $multitable 
  where field_name = 'homeroom' and userid = ?");
$sth->execute( $rec{userid} );
if ($DBI::errstr){ print $DBI::errstr; die $DBI::errstr;}

print qq{<tr style="background-color:#DDD;"><td class="ra">$lex{Current} $lex{'Homeroom(s)'}</td>\n<td>};
while ( my ( $id, $fieldvalue ) = $sth->fetchrow ) {
    print qq{ <input type="text" name="m:$id" style="width:6ch;" value="$fieldvalue">}
}
print qq{</td></tr>\n};

print qq{<tr><td class="ra">$lex{Add} $lex{'Homeroom(s)'}</td>};
print qq{<td><input type="text" name="m:addhomeroom" style="width:12ch;"> };
print qq{$lex{'Separate with Spaces'}</td></tr>\n};
print qq{</table>\n};


# Access Controls (typically Catalysts for Reading, Math)
print qq{<table cellpadding="3" cellspacing="0" border="0" };
print qq{style="margin:1em;padding:0.4em;border:1px solid gray;">\n};
print qq{<caption style="font-weight:bold;text-align:left;">};
print qq{Access Controls (typically Catalysts for Reading, Math)<br>};
print qq{values: DRA (reading),CMA(math),LLI(LLI)</caption>\n};

my $sth = $dbh->prepare("select id, field_value from $multitable 
  where field_name = 'access' and userid = ?");
$sth->execute( $rec{userid} );
if ($DBI::errstr){ print $DBI::errstr; die $DBI::errstr;}

print qq{<tr><td class="ra">$lex{Current} $lex{'Access Control(s)'}</td>\n<td>};
while ( my ( $id, $fieldvalue ) = $sth->fetchrow ) {
    print qq{ <input type="text" name="m:$id" style="width:12ch;" value="$fieldvalue">}
}
print qq{</td></tr>\n};

print qq{<tr><td class="ra">$lex{Add} $lex{'Access Control(s)'}</td><td><input type="text" };
print qq{name="m:addaccess" size="6"> $lex{'Separate with Spaces'}</td></tr>\n};
print qq{</table>\n};


# INAC Nominal Roll Job Description

if ( $fnmode ) {

    # Load existing occupations structure 
    eval $rec{'occupations'};
    if ( $@ ) {
	print $lex{Error}. " $@<br>\n";
	die $lex{Error}. " $@\n";
    }


    # Load FTE
    my $sth = $dbh->prepare("select defaultvalue from meta where tableid = ? and fieldid = ?");
    $sth->execute('student_inac','fte' );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my $default = $sth->fetchrow;
    my %fte = split(/\s+/, $default);
    foreach my $key ( keys %fte ) { $fte{$key} =~ s/\_/ /g; }


    # Load Grades
    $sth = $dbh->prepare("select distinct grade from student order by grade");
    $sth->execute;
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my @grades;
    while ( my $gr = $sth->fetchrow ) {
	if ( $gr eq 'EC' or $gr eq 'ECS' ) { $gr = 'K'; }
	if ( $gr eq 'K5' or $gr eq 'K4' ) { $gr = 'K'; }
	push @grades, $gr;
    }


    # Load OccupationTypes
    $sth = $dbh->prepare("select defaultvalue from meta where tableid = ? and fieldid = ?");
    $sth->execute('staff','occupations' );
    if ($DBI::errstr) { print $DBI::errstr; die $DBI::errstr; }
    my $default = $sth->fetchrow;
    my %jobtype = split(/\s+/, $default);
    foreach my $key ( keys %jobtype ) { $jobtype{$key} =~ s/\_/ /g; }


    print qq{<table cellspacing="0" cellpadding="3" border="1" style="margin:2em 0 0 1em;">\n};
    print qq{<caption style="font-size:120%;font-weight:bold;text-align:left;">};
    print qq{INAC Nominal Roll - Job Description</caption>\n};
    print qq{<tr style="background-color:#DDD;"><td class="bla">$lex{Occupation} $lex{Type}</td>};
    print qq{<td class="bla">$lex{Percent}<br>1-100</td></tr>\n};

    for my $i (1..$maxmulti ) {

	# Job Type
	my $type = $occ{$i}{'type'};
	print qq{<tr><td class="la"><select name="$i:type:">};
	print qq{<option value="$type">$jobtype{$type}</option>\n};
	foreach my $key (sort keys %jobtype ) { 
	    print qq{<option value="$key">$jobtype{$key}</option>}; 
	}
	print qq{<option></option></select></td>\n};

	# Percent
	my $val = $occ{$i}{'percent'};
	print qq{<td class="cn"><input type="text" name="$i:percent:" size="4" };
	print qq{value="$val"></td></tr>\n};
    }

    print qq{</table>\n};
    # end of Job Entry Values
    
} # end of first nations mode (fnmode)


print qq{<div style="margin:1em;">};
print qq{<input type="submit" value="$lex{Update} $lex{'Staff Member'}"></div>\n};
print qq{</form></body></html>\n};




#--------------
sub writeRecord {
#--------------

    # foreach my $key ( sort keys %arr ) { print qq{K:$key V:$arr{$key}<br>\n}; }


    use Data::Dumper;
    $Data::Dumper::Purity = 1;
    $Data::Dumper::Indent = 0;

    # Extract the Occupations Fields, if present
    my %occ; # occupation structure
    foreach my $key ( sort keys %arr) {
	my @vals = split(':', $key );
	if ( $vals[1] ) { # we have some stuff to take care of.
	    if ( $vals[0] eq 'm' ) { next; } # skip multirecs.
	    if ( $arr{$key} ) { # if a defined value, add to structure
		if ( $#vals == 2 ) {
		    $occ{"$vals[0]"}{"$vals[1]"}{"$vals[2]"} = $arr{$key};
		} else {
		    $occ{"$vals[0]"}{"$vals[1]"} = $arr{$key};
		}
	    }
	    delete $arr{$key};
	}
     }

    if ( %occ ) {
	my $value_ref = [ ];
	my $name_ref = [ ];

	$dataname = '*occ';
	push @$name_ref, $dataname;
	push @$value_ref, \%occ;

	my $d = Data::Dumper->new( $value_ref, $name_ref );
	my $occupations = $d->Dump;
	$arr{'occupations'} = $occupations;
    }

#    print qq{<br><div>After Occupations Removed</div>\n};
#    foreach my $key ( sort keys %arr ) { print qq{K:$key V:$arr{$key}<br>\n}; }
#    exit;
    

    # Check for any duplicate userid's in current records.
    my (%dup,%name);
    my $sth = $dbh->prepare("select userid, lastname, firstname from staff");
    $sth->execute;
    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr }
    while ( my ($userid, $ln, $fn)  = $sth->fetchrow ) {
	if ( $dup{$userid} ) { # we have a duplicate!
	    print qq{<h1>Duplicate Userid Found:$userid<br>\n};
	    print qq{1.$name{userid}  2. $ln, $fn</h1>\n};
	    print qq{</body></html>\n};
	    exit;	    
	} else { # add to hashes.
	    $dup{$userid} = 1;
	    $name{$userid} = qq{$ln, $fn};
	}
    }

    # Check if current userid WOULD make a duplicate. Done below.

    
    # Make sure we have userid, password
    if ( not $arr{userid} or not $arr{password} ){
	print qq{<h1>$lex{'You MUST have a unique userid and a password'}</h1>\n};
	print qq{</body></html>\n};
	exit;
    }


    # Check for password quality using Data::Password module.
    $MINLEN = $g_staffpwd_minlen;  # from admin.conf settings.
    $MAXLEN = $g_staffpwd_maxlen;
    $FOLLOWING = $g_staffpwd_following;
    $GROUPS = $g_staffpwd_groups;
    if ( $g_staffpwd_dictionary ) {
	$DICTIONARY = $g_staffpwd_dictionary;
    } else {
	$DICTIONARY = 0; # turn it off.
    }

    if ( IsBadPassword( $arr{password} )) {
	print qq{<h1 style="color:red;">$lex{Password} $lex{Error}<br>\n};
	print IsBadPassword( $arr{password} );
	print qq{</h1></body></html>\n};
	exit;
    }

    
    # Checkbox updates (since no returned values for these)
    my %checkboxfields;
    my $sth = $dbh->prepare("select distinct fieldid from meta 
      where tableid = 'staff' and formtype = 'checkbox'");
    $sth->execute;
    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr }
    while ( my $fieldid = $sth->fetchrow ) {
        $checkboxfields{$fieldid} = 1;
    }


    # Pre-registration mode or normal mode?
    my $table = 'staff';
    my $prereg;
    if ( $arr{tbl} ){ 
	$prereg = $arr{tbl};
	$table = 'prereg_staff';
    }
    delete $arr{tbl};

    
    # Check for identical userid, if record updated.
    $sth = $dbh->prepare("select lastname, firstname from $table where userid = ? and id != ?");
    $sth->execute( $arr{userid}, $arr{id} );
    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
    while ( my ($lastname, $firstname ) = $sth->fetchrow ) {
	print qq{<h1 style="color:red;">$lex{Error}: };
	print qq{$lex{'Userid Already Exists'}: $firstname $lastname</h1>\n};
	print qq{</body></html>\n};
	exit;
    }

    # View Passed values.
    # foreach my $key ( sort keys %arr) { print qq{K:$key V:$arr{$key}<br>\n}; }

    my $id = $arr{id};
    delete $arr{id};

    if ( $id ) { # we are updating existing records.

	# Load Current Record for Audit Function
	$sth = $dbh->prepare("select * from staff where id = ?");
	$sth->execute( $id );
	if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
	my $sref = $sth->fetchrow_hashref; # staff record reference

	my $curruserid = $sref->{userid};
	if ( $arr{userid} ne $curruserid ) {
	    print qq{<h3>Changes to a current Userid are not allowed</h3>\n};
	    $arr{userid} = $curruserid;
	}

	
	# Add Audit record.
	my %audit;
	$audit{userid} = $ENV{REMOTE_USER};
	$audit{ipaddr} = $ENV{REMOTE_ADDR};
	$audit{scriptname} = $self;
	$audit{tablename} = 'staff';
	$audit{tableid} = $id;
	$audit{startval} = $sref; 
	$audit{endval} = \%arr;

	addAudit( \%audit, $dbh );


	# Update staff table records
	foreach my $key ( keys %arr ) {
	    if ( $key =~ m/^m:/ ) { next; } # skip multirecord values
	    if ( not $arr{$key} ) { undef $arr{$key}; } # NULL records
	    $sth = $dbh->prepare("update $table set $key = ? where id = $id");

	    $sth->execute( $arr{$key} );

	    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
	}


	# Update any checkbox fields.
	foreach my $fieldid ( keys %checkboxfields ) {

	    # if field HAS a value, then make it NULL, since checkboxes not passed in forms.
	    # Load the field.
	    my $sth = $dbh->prepare("select $fieldid from staff where id = ?");
	    $sth->execute( $id );
	    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr }
	    my $recval = $sth->fetchrow;
	    my $passedval = $arr{$fieldid};

	    # Null field if present
	    if ( $recval and not $passedval ) { #record has 1, but nothing passed; we must fix.
		my $sth1 = $dbh->prepare("update staff set $fieldid = NULL where id = ?");
		$sth1->execute( $id );
		if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr }
	    }
	}


	
    } else {  # we are adding a record.....
	
	my %new = %arr;
	
	# remove multi values from %new;
	foreach my $key ( sort keys %new) {
	    my ($type,$field ) = split(':', $key );
	    if ( $type eq 'm' ) {
		delete $new{$key};
	    }
	}
	
	# foreach my $key ( sort keys %arr) { print "K:$key V:$arr{$key}<br>\n"; }

	my @values;
	my @fieldnames;

	# push in the passed values (skip blanks since an add process)
	foreach my $key ( sort keys %new ) { # run through all passed values.
	    if ( $new{$key} ) {    # if we have a value
		push @values, $new{$key};
		push @fieldnames, $key;
	    }
	}


	# Join values
	my $fields = join(',', @fieldnames );
	foreach my $val ( @values ) {
	    $val = $dbh->quote( $val );
	}
	my $values = join(',', @values );


	# print "Insert Fields: $fields  Values: $values<br>\n";
	my $sth = $dbh->prepare("insert into $table ( $fields ) values( $values )");
	$sth->execute;
	if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }

	# Insert a staff transfer record also.
	my $sth = $dbh->prepare("insert into staff_transfer 
          (userid, date, type, lastname, firstname, certificatenumber) 
          values(?,now(), 'enrol', ?,?,?)");
	$sth->execute($new{userid}, $new{lastname}, $new{firstname}, $new{certification1} );
	if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
	
	
    } # end of inserting a staff record; no multitable updates yet....


    
    # Update records in staff_multi
    my $sth2 = $dbh->prepare("select field_name from $multitable where id = ?");

    my $sth = $dbh->prepare("insert into $multitable ( userid, field_name, field_value ) 
         values( ?, ?, ? )");

    # parse multiple m:addgrade into a single one.
    my @grvals; # grade values.
    foreach my $key ( keys %arr ) {
	if ( $key =~ m/^m:addgrade/ ) { 
	    if ( $arr{$key} ) { # we have a grade value
		push @grvals, $arr{$key};
	    }
	    delete $arr{$key}; # delete even if blank value.
	}
    }
    if ( @grvals ) {
	my $gr = join(' ', @grvals);
	$arr{"m:addgrade"} = $gr;
    }
    # we have now parsed multiple addgrade vals into a single one

    
    foreach my $key ( keys %arr ) {
	if ( $key =~ m/^m:/ ) { 
	    my $id = $key;
	    $id =~ s/^m://; # strip off start.

	    if ( $id =~ m/add/ and $arr{$key} ) { # we are adding new multirecords...
		$id =~ s/^add//; # strip add
		my $fieldname = $id;
		if ( $fieldname eq 'position' ) { # Add single record
		    $sth->execute( $arr{userid}, 'position', $arr{$key} );
		    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }

		} else { # possible multiple records add (grades, homerooms)
		    foreach my $val ( split / /, $arr{$key} ) {
			$sth->execute( $arr{userid}, $fieldname, $val );
			if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
		    }
		}

	    } elsif ( not $arr{$key} ) { # delete the record, since remaining value is record id.
		my $sth = $dbh->prepare("delete from $multitable where id = ?");
		$sth->execute( $id );
		if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }

	    } else { # update the record;

		# Get Fieldname for the record.
		$sth2->execute( $id );
		if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
		my $fieldname = $sth2->fetchrow;

		# Check for spaces in value
		if ( $arr{$key} =~ m/\s+/ and ( $fieldname eq 'grade' or $fieldname eq 'homeroom' )) {
		    print qq{<h3>$lex{Error}: $lex{'Spaces Not Allowed'} - $arr{$key}</h3>\n};
		    print qq{</body></html>\n};
		    exit;
		}

		my $sth = $dbh->prepare("update $multitable set field_value = ? where id = ?");
		$sth->execute( $arr{$key}, $id );
		if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
	    }
	}
    } # end of multi table update.


    if ( not $DBI::errstr ) {
	print qq{<h3>$lex{'Record(s) Updated'}</h3>\n};

    } else {
	print qq{<h3>$lex{Error}: $DBI::errstr<br>$lex{Contact} };
	print qq{$adminname - <a href="mailto:$adminemail">$adminemail</a></h3>\n};
    }


    # Search for staff_multi REDUNDANT records and remove.
    my %multiusers;
    my $sth = $dbh->prepare("select distinct userid from staff_multi");
    $sth->execute;
    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
    while ( my $userid = $sth->fetchrow ) {
	$multiusers{$userid} = 1;
    }

    # remove current users from multiusers hash to leave redundant records
    my $sth = $dbh->prepare("select userid from staff");
    $sth->execute;
    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
    while ( my $userid = $sth->fetchrow ) {
	delete $multiusers{$userid};
    }

    # remove extra ones.
    if ( %multiusers ) {
	my $sth = $dbh->prepare("delete from staff_multi where userid = ?");
	foreach my $userid ( keys %multiusers ) {
	    print qq{Multi:$userid<br>\n};
	    $sth->execute( $userid );
	    if ( $DBI::errstr ) { print $DBI::errstr; die $DBI::errstr; }
	}
    }


    # Don't really need these....
    # print qq{<p>[ <a href="$homepage">$lex{Main}</a> |};
    # print qq{ <a href="$eoypage">$lex{Eoy}</a> ]</p>\n};

    if ( $prereg ) {
	print qq{[ <a href="staffdeled.pl?tbl=prereg">};
	print qq{$lex{Edit} <span style="color:red;">$lex{Preregistration}</span> $lex{'Staff Member'}};
    } else {
	print qq{[ <a href="staffdeled.pl">$lex{Edit} $lex{'Staff Member'}};
    }

    print qq{</a> ]</p></body></html>\n};

    exit;

}
